/**
 * SuperDev Help Button with Screenshot
 * Works on Windows, Mac, and Linux browsers
 * 
 * Usage:
 * 1. Include html2canvas: <script src="https://html2canvas.hertzen.com/dist/html2canvas.min.js"></script>
 * 2. Include this script: <script src="help-button.js"></script>
 * 3. Call initHelpButton('your-support@email.com') on page load
 */

function initHelpButton(supportEmail = 'support@macrotechtitan.com') {
    // Inject CSS
    const style = document.createElement('style');
    style.textContent = `
        .help-btn {
            position: fixed;
            bottom: 30px;
            right: 30px;
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            color: white;
            border: none;
            padding: 12px 20px;
            border-radius: 50px;
            font-size: 0.9rem;
            font-weight: 600;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 8px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
            transition: transform 0.2s, box-shadow 0.2s;
            z-index: 9999;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        }
        .help-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 25px rgba(0, 0, 0, 0.4);
        }
        .help-btn svg { width: 18px; height: 18px; }
        .help-modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            z-index: 10000;
            justify-content: center;
            align-items: center;
        }
        .help-modal.active { display: flex; }
        .help-modal-content {
            background: white;
            border-radius: 16px;
            padding: 30px;
            max-width: 500px;
            width: 90%;
            text-align: center;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        }
        .help-modal h3 { margin-bottom: 15px; color: #1f2937; }
        .help-modal p { color: #6b7280; margin-bottom: 20px; font-size: 0.9rem; }
        .help-modal-buttons { display: flex; gap: 10px; justify-content: center; flex-wrap: wrap; }
        .help-modal-btn {
            padding: 12px 24px;
            border-radius: 10px;
            font-weight: 600;
            cursor: pointer;
            border: none;
            font-size: 0.95rem;
        }
        .help-modal-btn.primary {
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            color: white;
        }
        .help-modal-btn.secondary {
            background: #f3f4f6;
            color: #374151;
        }
    `;
    document.head.appendChild(style);

    // Inject HTML
    const html = `
        <button class="help-btn" id="helpBtn">
            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8.228 9c.549-1.165 2.03-2 3.772-2 2.21 0 4 1.343 4 3 0 1.4-1.278 2.575-3.006 2.907-.542.104-.994.54-.994 1.093m0 3h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
            </svg>
            Help
        </button>
        <div class="help-modal" id="helpModal">
            <div class="help-modal-content">
                <h3>Need Help?</h3>
                <p>Click the button below to capture a screenshot and open your email client. The screenshot will be downloaded - please attach it to your email.</p>
                <div class="help-modal-buttons">
                    <button class="help-modal-btn primary" id="captureBtn">Capture & Email Support</button>
                    <button class="help-modal-btn secondary" id="cancelBtn">Cancel</button>
                </div>
            </div>
        </div>
    `;
    document.body.insertAdjacentHTML('beforeend', html);

    // Event listeners
    const modal = document.getElementById('helpModal');
    
    document.getElementById('helpBtn').addEventListener('click', () => {
        modal.classList.add('active');
    });

    document.getElementById('cancelBtn').addEventListener('click', () => {
        modal.classList.remove('active');
    });

    modal.addEventListener('click', (e) => {
        if (e.target === modal) modal.classList.remove('active');
    });

    document.getElementById('captureBtn').addEventListener('click', async () => {
        modal.style.display = 'none';
        
        try {
            if (typeof html2canvas === 'undefined') {
                throw new Error('html2canvas not loaded');
            }
            
            const canvas = await html2canvas(document.body, {
                scale: 1,
                useCORS: true,
                allowTaint: true,
                scrollY: -window.scrollY
            });
            
            const link = document.createElement('a');
            const timestamp = new Date().toISOString().replace(/[:.]/g, '-');
            link.download = `screenshot-${timestamp}.png`;
            link.href = canvas.toDataURL('image/png');
            link.click();
            
            setTimeout(() => {
                const subject = encodeURIComponent('Support Request');
                const body = encodeURIComponent(`Hello Support Team,

I need help.

[Please describe your issue here]

---
Screenshot: Please attach the downloaded screenshot (screenshot-${timestamp}.png)
Browser: ${navigator.userAgent}
URL: ${window.location.href}
Time: ${new Date().toLocaleString()}
`);
                window.location.href = `mailto:${supportEmail}?subject=${subject}&body=${body}`;
            }, 500);
            
            alert('Screenshot downloaded! Please attach it to the email that will open.');
        } catch (error) {
            alert('Could not capture screenshot. Please take a manual screenshot and email ' + supportEmail);
            window.location.href = `mailto:${supportEmail}?subject=Support%20Request`;
        }
        
        modal.classList.remove('active');
        modal.style.display = '';
    });
}

// Auto-initialize if support email is set via data attribute
document.addEventListener('DOMContentLoaded', () => {
    const script = document.currentScript || document.querySelector('script[data-support-email]');
    if (script && script.dataset.supportEmail) {
        initHelpButton(script.dataset.supportEmail);
    }
});
